
suppressMessages(library(RKernel))
options(width=120)

# ## Preparing the data
# The data we are using are in the file "ESS-relig-CHES.RData".

suppressMessages(library(memisc))
library(magrittr)


load(file="ESS-relig-CHES.RData")


load(file="perc-denom-country-wide.RData")


ESS_work_west <- subset(ESS_relatt_denom_CHES,
    country2 %in% c(
        "Austria",
        "Belgium/Wallonia","Belgium/Flanders","Belgium/Brussels",
        "Denmark",
        "France",
        "Germany (West)","Germany/Bavaria",
        "Ireland",
        "Italy",
        "Luxembourg",
        "Netherlands",
        "Norway",
        "Portugal",
        "Spain/Castilia","Spain/Catalunia","Spain/Basque",
        "Sweden",
        "Switzerland",
        "United Kingdom/England",
        "United Kingdom/Scotland",
        "United Kingdom/Wales"
    ) & !is.na(galtan),
    select=c(CHES_id,
             year,
             essround,
             denom,
             relatt,
             oesch8,
             wgt,
             id,
             country2,
             lrecon,
             galtan,
             spendvtax,
             deregulation,
             redistribution,
             civlib_laworder,
             sociallifestyle,
             religious_principles,
             immigration))


ESS_work_west %<>% as.data.frame()

# The information about religious denomination appears to be incorrect or misleading 
# in France until 2005 and in the United Kingdom from 2004 to 2007 and after 2020.

options(contrasts=c("contr.sum","contr.sum"))


ESS_work_west %<>% within({
    Country <- recode(as.character(country2),
                      c("Belgium/Wallonia","Belgium/Flanders","Belgium/Brussels") -> "Belgium",
                      c("Germany (West)","Germany/Bavaria") -> "Germany",
                      c("Spain/Castilia","Spain/Catalunia","Spain/Basque") -> "Spain",
                      c("United Kingdom/England","United Kingdom/Scotland","United Kingdom/Wales") 
                          -> "United Kingdom",
                      copy=TRUE)
    Country <- factor(Country,levels=sort(levels(Country)))
    denom[Country=="France" & year <= 2005 | 
         Country=="United Kingdom" & year >= 2004 & year <= 2007] <- NA
    cathprot <- recode(as.numeric(denom),
                      "Catholic"     <- 1,
                      "Protestant"   <- 2
                     )
})


ESS_work_west <- merge(ESS_work_west, perc.denom.country.wide,
                       by.x = 'Country',
                       by.y = 'country',
                       all.x = TRUE,
                       all.y = FALSE)


ESS_work_west %<>% within({
    foreach(x = c(lrecon,
                  galtan,
                  spendvtax,
                  deregulation,
                  redistribution,
                  civlib_laworder,
                  sociallifestyle,
                  religious_principles,
                  immigration),
            y = c(lrecon.sq,
                  galtan.sq,
                  spendvtax.sq,
                  deregulation.sq,
                  redistribution.sq,
                  civlib_laworder.sq,
                  sociallifestyle.sq,
                  religious_principles.sq,
                  immigration.sq),
            {
                x <- (x-5)/10
                y <- x^2
            })
    oesch8 <- factor(oesch8,levels=levels(oesch8)[c(3,1:2,4:8)]) # Production workers as baseline
    country <- factor(country2) # Drop countries not included in this subset
    time <- (year - 2010)/10
    contrasts(cathprot) <- contr("sum", base = 1)
    contrasts(oesch8) <- contr("sum", base = 1)
    contrasts(relatt) <- contr("sum", base = 1)
    No_religion <- (No_religion - 50)/100
    Denom_Het <- 2*(1-(Cath_Prot/100)^2-(1-Cath_Prot/100)^2)
    })

# ## Model estimation

library(mclogit)


model_galtan <- mclogit(wgt|id~galtan.sq+
                                lrecon.sq+
                                galtan/(oesch8+relatt*cathprot)*time+
                                galtan/(relatt)*(No_religion+Cath_Prot+Denom_Het)+
                                lrecon/(oesch8+relatt+cathprot)*time
                                -time,
                                random=~1|CHES_id/essround,
                                data=ESS_work_west,
                                 trace=TRUE,
                                estimator="REML"
                                )


# ## Wald tests

source("Wald.R")


(Wald_galtan_full <- 
 Wald(model_galtan,
     list(~galtan.sq,
          ~galtan,
          ~galtan:time,
          ~galtan:relatt,
          ~galtan:relatt:time,
          ~galtan:cathprot,
          ~galtan:cathprot:time,
          ~galtan:relatt:cathprot,
          ~galtan:relatt:cathprot:time,
          ~galtan:No_religion,
          ~galtan:relatt:No_religion,
          ~galtan:Cath_Prot,
          ~galtan:relatt:Cath_Prot,
          ~galtan:Denom_Het,
          ~galtan:relatt:Denom_Het
           )))


Wald_galtan_full %>%
        dimrename(          
        "galtan.sq" = "GAL--TAN squared",
        "galtan" = "GAL--TAN",
        "galtan:time" = "$\\times$ time",
        "galtan:relatt" = "$\\times$ church attendance",
        "galtan:relatt:time" = "$\\times$ church attendance $\\times$ time",
        "galtan:cathprot" = "$\\times$ Catholic/Protestant",
        "galtan:cathprot:time" = "$\\times$ time $\\times$ Catholic/Protestant",
        "galtan:relatt:cathprot" = "$\\times$ church attendance $\\times$ Catholic/Protestant",
        "galtan:relatt:cathprot:time" = "$\\times$ church attendance $\\times$ Catholic/Protestant $\\times$ time",
        "galtan:No_religion" = "$\\times$ secular",
        "galtan:relatt:No_religion" = "$\\times$ church attendance $\\times$ secular",
        "galtan:Cath_Prot" = "$\\times$ denom. balance",
        "galtan:relatt:Cath_Prot" = "$\\times$ church attendance $\\times$ denom. balance",
        "galtan:Denom_Het" = "$\\times$ denom. heterogeneity",
        "galtan:relatt:Denom_Het" = "$\\times$ church attendance $\\times$ denom. heterogeneity",
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="churchat-context-galtan/Wald-churchat-context-galtan-full.tex")


(Wald_galtan_short <- 
 Wald(model_galtan,
     list(~galtan.sq,
          ~galtan,
          ~galtan:time,
          ~galtan:relatt,
          ~galtan:relatt:time,
          ~galtan:relatt:cathprot,
          ~galtan:relatt:cathprot:time
           )))


Wald_galtan_short %>%
        dimrename(          
        "galtan.sq" = "GAL--TAN squared",
        "galtan" = "GAL--TAN",
        "galtan:time" = "$\\times$ time",
        "galtan:relatt" = "$\\times$ church attendance",
        "galtan:relatt:time" = "$\\times$ church attendance $\\times$ time",
        "galtan:relatt:cathprot" = "$\\times$ church attendance $\\times$ Catholic/Protestant",
        "galtan:relatt:cathprot:time" = "$\\times$ church attendance $\\times$ Catholic/Protestant $\\times$ time",
        ) %>% rename(
            "Chisq"="$W$",
            "Pr(>Chisq)"="$p$-value") %>%
        toLatex(digits=c(1,0,3)) %>% writeLines(con="churchat-context-galtan/Wald-churchat-context-galtan-short.tex")


save(model_galtan,file="model-churchat-cathprot-galtan-context.RData")

